const express = require('express');
const sqlite3 = require('sqlite3').verbose();
const bcrypt = require('bcryptjs');
const session = require('express-session');
const bodyParser = require('body-parser');
const nodemailer = require('nodemailer');
const path = require('path');

const app = express();
const PORT = process.env.PORT || 3003;

// ミドルウェア設定
app.use(bodyParser.urlencoded({ extended: true }));
app.use(bodyParser.json());
app.use(express.static('public'));
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

// セッション設定
app.use(session({
    secret: 'reservation-system-secret',
    resave: false,
    saveUninitialized: false,
    cookie: { secure: false }
}));

// SQLiteデータベース接続
const db = new sqlite3.Database('./reservation.db');

// データベース初期化
db.serialize(() => {
    // 施設テーブル
    db.run(`CREATE TABLE IF NOT EXISTS facilities (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )`);

    // スケジュールテーブル
    db.run(`CREATE TABLE IF NOT EXISTS schedules (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        facility_id INTEGER NOT NULL,
        date DATE NOT NULL,
        start_time TIME NOT NULL,
        end_time TIME NOT NULL,
        max_capacity INTEGER DEFAULT 10,
        current_reservations INTEGER DEFAULT 0,
        is_available BOOLEAN DEFAULT TRUE,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (facility_id) REFERENCES facilities(id)
    )`);

    // 予約テーブル
    db.run(`CREATE TABLE IF NOT EXISTS reservations (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        schedule_id INTEGER NOT NULL,
        name TEXT NOT NULL,
        name_kana TEXT NOT NULL,
        email TEXT NOT NULL,
        age INTEGER NOT NULL,
        education TEXT NOT NULL,
        motivation TEXT NOT NULL,
        other_notes TEXT,
        status TEXT DEFAULT 'pending',
        reservation_number TEXT UNIQUE,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (schedule_id) REFERENCES schedules(id)
    )`);

    // 管理者テーブル
    db.run(`CREATE TABLE IF NOT EXISTS admins (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        username TEXT UNIQUE NOT NULL,
        password_hash TEXT NOT NULL,
        email TEXT NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )`);

    // 初期データ挿入
    db.get("SELECT COUNT(*) as count FROM facilities", (err, row) => {
        if (row.count === 0) {
            const facilities = [
                'あさくら苑 / あさくら苑デイサービスセンター',
                'あさくら苑デイサービスセンター',
                'リバティーハウスあさくら'
            ];
            
            facilities.forEach(facility => {
                db.run("INSERT INTO facilities (name) VALUES (?)", [facility]);
            });
        }
    });

    // 管理者アカウント作成
    db.get("SELECT COUNT(*) as count FROM admins", (err, row) => {
        if (row.count === 0) {
            const hashedPassword = bcrypt.hashSync('admin123', 10);
            db.run("INSERT INTO admins (username, password_hash, email) VALUES (?, ?, ?)", 
                ['admin', hashedPassword, 'admin@example.com']);
        }
    });

    // サンプルスケジュール
    db.get("SELECT COUNT(*) as count FROM schedules", (err, row) => {
        if (row.count === 0) {
            const schedules = [
                [1, '2025-07-27', '09:00:00', '10:00:00', 10],
                [1, '2025-07-27', '10:00:00', '11:00:00', 10],
                [1, '2025-07-27', '11:00:00', '12:05:00', 10],
                [2, '2025-07-28', '09:00:00', '10:00:00', 8],
                [2, '2025-07-28', '10:00:00', '11:00:00', 8],
                [3, '2025-07-29', '09:00:00', '10:00:00', 12],
                [3, '2025-07-29', '10:00:00', '11:00:00', 12]
            ];
            
            schedules.forEach(schedule => {
                db.run("INSERT INTO schedules (facility_id, date, start_time, end_time, max_capacity) VALUES (?, ?, ?, ?, ?)", 
                    schedule);
            });
        }
    });
});

// メール設定
const transporter = nodemailer.createTransport({
    host: 'smtp.gmail.com',
    port: 587,
    secure: false,
    auth: {
        user: process.env.EMAIL_USER || 'your-email@gmail.com',
        pass: process.env.EMAIL_PASS || 'your-password'
    }
});

// 予約番号生成関数
function generateReservationNumber() {
    return Date.now().toString().slice(-8);
}

// ルート: トップページ
app.get('/', (req, res) => {
    db.all('SELECT * FROM facilities ORDER BY id', (err, facilities) => {
        if (err) {
            console.error('施設取得エラー:', err);
            return res.status(500).send('サーバーエラー');
        }
        res.render('index', { facilities });
    });
});

// ルート: 施設選択後のカレンダー取得
app.get('/api/schedules/:facilityId', (req, res) => {
    const facilityId = req.params.facilityId;
    const year = req.query.year || new Date().getFullYear();
    const month = req.query.month || (new Date().getMonth() + 1);
    
    const query = `
        SELECT s.*, f.name as facility_name 
        FROM schedules s 
        JOIN facilities f ON s.facility_id = f.id 
        WHERE s.facility_id = ? 
        AND strftime('%Y', s.date) = ? 
        AND strftime('%m', s.date) = ?
        AND s.is_available = 1
        AND s.current_reservations < s.max_capacity
        ORDER BY s.date, s.start_time
    `;
    
    db.all(query, [facilityId, year, month.toString().padStart(2, '0')], (err, schedules) => {
        if (err) {
            console.error('スケジュール取得エラー:', err);
            return res.status(500).json({ error: 'サーバーエラー' });
        }
        res.json(schedules);
    });
});

// ルート: 予約処理
app.post('/api/reserve', (req, res) => {
    const { scheduleId, name, nameKana, email, age, education, motivation, otherNotes } = req.body;
    
    // バリデーション
    if (!scheduleId || !name || !nameKana || !email || !age || !education || !motivation) {
        return res.status(400).json({ error: '必須項目を入力してください' });
    }
    
    const reservationNumber = generateReservationNumber();
    
    db.run(`INSERT INTO reservations 
        (schedule_id, name, name_kana, email, age, education, motivation, other_notes, reservation_number)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [scheduleId, name, nameKana, email, age, education, motivation, otherNotes, reservationNumber],
        function(err) {
            if (err) {
                console.error('予約挿入エラー:', err);
                return res.status(500).json({ error: 'サーバーエラー' });
            }
            
            // 現在の予約数を更新
            db.run('UPDATE schedules SET current_reservations = current_reservations + 1 WHERE id = ?', [scheduleId]);
            
            // 申込者にメール送信
            sendConfirmationEmail(email, name, reservationNumber, scheduleId);
            
            res.json({ 
                success: true, 
                reservationNumber: reservationNumber,
                message: '予約申込が完了しました。承認をお待ちください。'
            });
        });
});

// 確認メール送信
function sendConfirmationEmail(email, name, reservationNumber, scheduleId) {
    const query = `
        SELECT s.*, f.name as facility_name 
        FROM schedules s 
        JOIN facilities f ON s.facility_id = f.id 
        WHERE s.id = ?
    `;
    
    db.get(query, [scheduleId], (err, schedule) => {
        if (err || !schedule) return;
        
        const mailOptions = {
            from: process.env.EMAIL_USER,
            to: email,
            subject: '説明会予約申込確認',
            html: `
                <h2>説明会予約申込を受け付けました</h2>
                <p>${name}様</p>
                <p>この度は説明会へのお申込みありがとうございます。</p>
                
                <h3>予約詳細</h3>
                <ul>
                    <li><strong>予約番号:</strong> ${reservationNumber}</li>
                    <li><strong>施設:</strong> ${schedule.facility_name}</li>
                    <li><strong>日時:</strong> ${schedule.date} ${schedule.start_time} - ${schedule.end_time}</li>
                </ul>
                
                <p>管理者による承認後、確定メールをお送りいたします。</p>
                <p>ご不明な点がございましたら、お気軽にお問い合わせください。</p>
            `
        };
        
        transporter.sendMail(mailOptions, (error, info) => {
            if (error) {
                console.error('メール送信エラー:', error);
            } else {
                console.log('確認メール送信完了:', info.response);
            }
        });
    });
}

// 管理者ログイン
app.get('/admin', (req, res) => {
    if (req.session.adminId) {
        return res.redirect('/admin/dashboard');
    }
    res.render('admin-login');
});

app.post('/admin/login', (req, res) => {
    const { username, password } = req.body;
    
    db.get('SELECT * FROM admins WHERE username = ?', [username], (err, admin) => {
        if (err || !admin) {
            return res.render('admin-login', { error: 'ユーザー名またはパスワードが間違っています' });
        }
        
        bcrypt.compare(password, admin.password_hash, (compareErr, isMatch) => {
            if (compareErr || !isMatch) {
                return res.render('admin-login', { error: 'ユーザー名またはパスワードが間違っています' });
            }
            
            req.session.adminId = admin.id;
            res.redirect('/admin/dashboard');
        });
    });
});

// 管理者ダッシュボード
app.get('/admin/dashboard', (req, res) => {
    if (!req.session.adminId) {
        return res.redirect('/admin');
    }
    
    const query = `
        SELECT r.*, s.date, s.start_time, s.end_time, f.name as facility_name
        FROM reservations r
        JOIN schedules s ON r.schedule_id = s.id
        JOIN facilities f ON s.facility_id = f.id
        ORDER BY r.created_at DESC
    `;
    
    db.all(query, (err, reservations) => {
        if (err) {
            console.error('予約取得エラー:', err);
            return res.status(500).send('サーバーエラー');
        }
        res.render('admin-dashboard', { reservations });
    });
});

// スケジュール管理画面
app.get('/admin/schedules', (req, res) => {
    if (!req.session.adminId) {
        return res.redirect('/admin');
    }
    
    const schedulesQuery = `
        SELECT s.*, f.name as facility_name 
        FROM schedules s 
        JOIN facilities f ON s.facility_id = f.id 
        ORDER BY s.date, s.start_time
    `;
    
    const facilitiesQuery = 'SELECT * FROM facilities ORDER BY id';
    
    db.all(schedulesQuery, (err, schedules) => {
        if (err) {
            console.error('スケジュール取得エラー:', err);
            return res.status(500).send('サーバーエラー');
        }
        
        db.all(facilitiesQuery, (err, facilities) => {
            if (err) {
                console.error('施設取得エラー:', err);
                return res.status(500).send('サーバーエラー');
            }
            
            res.render('admin-schedules', { schedules, facilities });
        });
    });
});

// スケジュール追加
app.post('/admin/schedules/add', (req, res) => {
    if (!req.session.adminId) {
        return res.status(401).json({ error: '認証が必要です' });
    }
    
    const { facilityId, date, startTime, endTime, maxCapacity } = req.body;
    
    if (!facilityId || !date || !startTime || !endTime || !maxCapacity) {
        return res.status(400).json({ error: '必須項目を入力してください' });
    }
    
    const query = `
        INSERT INTO schedules (facility_id, date, start_time, end_time, max_capacity)
        VALUES (?, ?, ?, ?, ?)
    `;
    
    db.run(query, [facilityId, date, startTime, endTime, maxCapacity], function(err) {
        if (err) {
            console.error('スケジュール追加エラー:', err);
            return res.status(500).json({ error: 'サーバーエラー' });
        }
        
        res.json({ success: true, message: 'スケジュールを追加しました' });
    });
});

// スケジュール削除
app.delete('/admin/schedules/:id', (req, res) => {
    if (!req.session.adminId) {
        return res.status(401).json({ error: '認証が必要です' });
    }
    
    const scheduleId = req.params.id;
    
    // 予約があるかチェック
    db.get('SELECT COUNT(*) as count FROM reservations WHERE schedule_id = ?', [scheduleId], (err, result) => {
        if (err) {
            console.error('予約確認エラー:', err);
            return res.status(500).json({ error: 'サーバーエラー' });
        }
        
        if (result.count > 0) {
            return res.status(400).json({ error: '予約が入っているスケジュールは削除できません' });
        }
        
        db.run('DELETE FROM schedules WHERE id = ?', [scheduleId], (err) => {
            if (err) {
                console.error('スケジュール削除エラー:', err);
                return res.status(500).json({ error: 'サーバーエラー' });
            }
            
            res.json({ success: true, message: 'スケジュールを削除しました' });
        });
    });
});

// 予約承認/拒否
app.post('/admin/approve/:id', (req, res) => {
    if (!req.session.adminId) {
        return res.status(401).json({ error: '認証が必要です' });
    }
    
    const reservationId = req.params.id;
    const { action } = req.body; // 'approved' or 'rejected'
    
    db.run('UPDATE reservations SET status = ? WHERE id = ?', [action, reservationId], (err) => {
        if (err) {
            console.error('承認処理エラー:', err);
            return res.status(500).json({ error: 'サーバーエラー' });
        }
        
        // 承認メール送信
        if (action === 'approved') {
            sendApprovalEmail(reservationId);
        }
        
        res.json({ success: true, message: '処理が完了しました' });
    });
});

// 承認メール送信
function sendApprovalEmail(reservationId) {
    const query = `
        SELECT r.*, s.date, s.start_time, s.end_time, f.name as facility_name
        FROM reservations r
        JOIN schedules s ON r.schedule_id = s.id
        JOIN facilities f ON s.facility_id = f.id
        WHERE r.id = ?
    `;
    
    db.get(query, [reservationId], (err, reservation) => {
        if (err || !reservation) return;
        
        const mailOptions = {
            from: process.env.EMAIL_USER,
            to: reservation.email,
            subject: '説明会予約確定のお知らせ',
            html: `
                <h2>説明会予約が確定しました</h2>
                <p>${reservation.name}様</p>
                <p>この度は説明会へのお申込みありがとうございます。あなたの予約が確定しました。</p>
                
                <h3>予約詳細</h3>
                <ul>
                    <li><strong>予約番号:</strong> ${reservation.reservation_number}</li>
                    <li><strong>施設:</strong> ${reservation.facility_name}</li>
                    <li><strong>日時:</strong> ${reservation.date} ${reservation.start_time} - ${reservation.end_time}</li>
                </ul>
                
                <p>当日は時間に余裕を持ってお越しください。</p>
                <p>ご不明な点がございましたら、お気軽にお問い合わせください。</p>
            `
        };
        
        transporter.sendMail(mailOptions, (error, info) => {
            if (error) {
                console.error('承認メール送信エラー:', error);
            } else {
                console.log('承認メール送信完了:', info.response);
            }
        });
    });
}

// 管理者ログアウト
app.get('/admin/logout', (req, res) => {
    req.session.destroy();
    res.redirect('/admin');
});

app.listen(PORT, () => {
    console.log(`サーバーがポート${PORT}で起動しました`);
    console.log(`ユーザー画面: http://localhost:${PORT}`);
    console.log(`管理者画面: http://localhost:${PORT}/admin`);
});