const express = require('express');
const mysql = require('mysql2');
const bcrypt = require('bcryptjs');
const session = require('express-session');
const bodyParser = require('body-parser');
const nodemailer = require('nodemailer');
const path = require('path');

const app = express();
const PORT = process.env.PORT || 3001;

// ミドルウェア設定
app.use(bodyParser.urlencoded({ extended: true }));
app.use(bodyParser.json());
app.use(express.static('public'));
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

// セッション設定
app.use(session({
    secret: 'reservation-system-secret',
    resave: false,
    saveUninitialized: false,
    cookie: { secure: false }
}));

// データベース接続
const db = mysql.createConnection({
    host: 'localhost',
    user: 'root',
    password: '',
    database: 'reservation_system'
});

db.connect((err) => {
    if (err) {
        console.error('データベース接続エラー:', err);
        console.log('MySQLが起動していない可能性があります。データベース機能は無効になります。');
    } else {
        console.log('データベースに接続されました');
    }
});

// メール設定
const transporter = nodemailer.createTransport({
    host: 'smtp.gmail.com',
    port: 587,
    secure: false,
    auth: {
        user: process.env.EMAIL_USER || 'your-email@gmail.com',
        pass: process.env.EMAIL_PASS || 'your-password'
    }
});

// 予約番号生成関数
function generateReservationNumber() {
    return Date.now().toString().slice(-8);
}

// ルート: トップページ
app.get('/', (req, res) => {
    const query = 'SELECT * FROM facilities ORDER BY id';
    db.query(query, (err, facilities) => {
        if (err) {
            console.error('施設取得エラー:', err);
            return res.status(500).send('サーバーエラー');
        }
        res.render('index', { facilities });
    });
});

// ルート: 施設選択後のカレンダー取得
app.get('/api/schedules/:facilityId', (req, res) => {
    const facilityId = req.params.facilityId;
    const year = req.query.year || new Date().getFullYear();
    const month = req.query.month || (new Date().getMonth() + 1);
    
    const query = `
        SELECT s.*, f.name as facility_name 
        FROM schedules s 
        JOIN facilities f ON s.facility_id = f.id 
        WHERE s.facility_id = ? 
        AND YEAR(s.date) = ? 
        AND MONTH(s.date) = ?
        AND s.is_available = TRUE
        AND s.current_reservations < s.max_capacity
        ORDER BY s.date, s.start_time
    `;
    
    db.query(query, [facilityId, year, month], (err, schedules) => {
        if (err) {
            console.error('スケジュール取得エラー:', err);
            return res.status(500).json({ error: 'サーバーエラー' });
        }
        res.json(schedules);
    });
});

// ルート: 予約フォーム表示
app.get('/reserve/:scheduleId', (req, res) => {
    const scheduleId = req.params.scheduleId;
    const query = `
        SELECT s.*, f.name as facility_name 
        FROM schedules s 
        JOIN facilities f ON s.facility_id = f.id 
        WHERE s.id = ?
    `;
    
    db.query(query, [scheduleId], (err, results) => {
        if (err || results.length === 0) {
            return res.status(404).send('スケジュールが見つかりません');
        }
        res.render('reservation-form', { schedule: results[0] });
    });
});

// ルート: 予約処理
app.post('/api/reserve', (req, res) => {
    const { scheduleId, name, nameKana, email, age, education, motivation, otherNotes } = req.body;
    
    // バリデーション
    if (!scheduleId || !name || !nameKana || !email || !age || !education || !motivation) {
        return res.status(400).json({ error: '必須項目を入力してください' });
    }
    
    const reservationNumber = generateReservationNumber();
    
    const insertQuery = `
        INSERT INTO reservations 
        (schedule_id, name, name_kana, email, age, education, motivation, other_notes, reservation_number)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;
    
    db.query(insertQuery, [scheduleId, name, nameKana, email, age, education, motivation, otherNotes, reservationNumber], (err, result) => {
        if (err) {
            console.error('予約挿入エラー:', err);
            return res.status(500).json({ error: 'サーバーエラー' });
        }
        
        // 現在の予約数を更新
        const updateQuery = 'UPDATE schedules SET current_reservations = current_reservations + 1 WHERE id = ?';
        db.query(updateQuery, [scheduleId], (updateErr) => {
            if (updateErr) {
                console.error('予約数更新エラー:', updateErr);
            }
        });
        
        // 申込者にメール送信
        sendConfirmationEmail(email, name, reservationNumber, scheduleId);
        
        res.json({ 
            success: true, 
            reservationNumber: reservationNumber,
            message: '予約申込が完了しました。承認をお待ちください。'
        });
    });
});

// 確認メール送信
function sendConfirmationEmail(email, name, reservationNumber, scheduleId) {
    const getScheduleQuery = `
        SELECT s.*, f.name as facility_name 
        FROM schedules s 
        JOIN facilities f ON s.facility_id = f.id 
        WHERE s.id = ?
    `;
    
    db.query(getScheduleQuery, [scheduleId], (err, results) => {
        if (err || results.length === 0) return;
        
        const schedule = results[0];
        const mailOptions = {
            from: process.env.EMAIL_USER,
            to: email,
            subject: '説明会予約申込確認',
            html: `
                <h2>説明会予約申込を受け付けました</h2>
                <p>${name}様</p>
                <p>この度は説明会へのお申込みありがとうございます。</p>
                
                <h3>予約詳細</h3>
                <ul>
                    <li><strong>予約番号:</strong> ${reservationNumber}</li>
                    <li><strong>施設:</strong> ${schedule.facility_name}</li>
                    <li><strong>日時:</strong> ${schedule.date} ${schedule.start_time} - ${schedule.end_time}</li>
                </ul>
                
                <p>管理者による承認後、確定メールをお送りいたします。</p>
                <p>ご不明な点がございましたら、お気軽にお問い合わせください。</p>
            `
        };
        
        transporter.sendMail(mailOptions, (error, info) => {
            if (error) {
                console.error('メール送信エラー:', error);
            } else {
                console.log('確認メール送信完了:', info.response);
            }
        });
    });
}

// 管理者ログイン
app.get('/admin', (req, res) => {
    if (req.session.adminId) {
        return res.redirect('/admin/dashboard');
    }
    res.render('admin-login');
});

app.post('/admin/login', (req, res) => {
    const { username, password } = req.body;
    
    const query = 'SELECT * FROM admins WHERE username = ?';
    db.query(query, [username], (err, results) => {
        if (err || results.length === 0) {
            return res.render('admin-login', { error: 'ユーザー名またはパスワードが間違っています' });
        }
        
        const admin = results[0];
        bcrypt.compare(password, admin.password_hash, (compareErr, isMatch) => {
            if (compareErr || !isMatch) {
                return res.render('admin-login', { error: 'ユーザー名またはパスワードが間違っています' });
            }
            
            req.session.adminId = admin.id;
            res.redirect('/admin/dashboard');
        });
    });
});

// 管理者ダッシュボード
app.get('/admin/dashboard', (req, res) => {
    if (!req.session.adminId) {
        return res.redirect('/admin');
    }
    
    const query = `
        SELECT r.*, s.date, s.start_time, s.end_time, f.name as facility_name
        FROM reservations r
        JOIN schedules s ON r.schedule_id = s.id
        JOIN facilities f ON s.facility_id = f.id
        ORDER BY r.created_at DESC
    `;
    
    db.query(query, (err, reservations) => {
        if (err) {
            console.error('予約取得エラー:', err);
            return res.status(500).send('サーバーエラー');
        }
        res.render('admin-dashboard', { reservations });
    });
});

// 予約承認/拒否
app.post('/admin/approve/:id', (req, res) => {
    if (!req.session.adminId) {
        return res.status(401).json({ error: '認証が必要です' });
    }
    
    const reservationId = req.params.id;
    const { action } = req.body; // 'approved' or 'rejected'
    
    const updateQuery = 'UPDATE reservations SET status = ? WHERE id = ?';
    db.query(updateQuery, [action, reservationId], (err, result) => {
        if (err) {
            console.error('承認処理エラー:', err);
            return res.status(500).json({ error: 'サーバーエラー' });
        }
        
        // 承認メール送信
        if (action === 'approved') {
            sendApprovalEmail(reservationId);
        }
        
        res.json({ success: true, message: '処理が完了しました' });
    });
});

// 承認メール送信
function sendApprovalEmail(reservationId) {
    const query = `
        SELECT r.*, s.date, s.start_time, s.end_time, f.name as facility_name
        FROM reservations r
        JOIN schedules s ON r.schedule_id = s.id
        JOIN facilities f ON s.facility_id = f.id
        WHERE r.id = ?
    `;
    
    db.query(query, [reservationId], (err, results) => {
        if (err || results.length === 0) return;
        
        const reservation = results[0];
        const mailOptions = {
            from: process.env.EMAIL_USER,
            to: reservation.email,
            subject: '説明会予約確定のお知らせ',
            html: `
                <h2>説明会予約が確定しました</h2>
                <p>${reservation.name}様</p>
                <p>この度は説明会へのお申込みありがとうございます。あなたの予約が確定しました。</p>
                
                <h3>予約詳細</h3>
                <ul>
                    <li><strong>予約番号:</strong> ${reservation.reservation_number}</li>
                    <li><strong>施設:</strong> ${reservation.facility_name}</li>
                    <li><strong>日時:</strong> ${reservation.date} ${reservation.start_time} - ${reservation.end_time}</li>
                </ul>
                
                <p>当日は時間に余裕を持ってお越しください。</p>
                <p>ご不明な点がございましたら、お気軽にお問い合わせください。</p>
            `
        };
        
        transporter.sendMail(mailOptions, (error, info) => {
            if (error) {
                console.error('承認メール送信エラー:', error);
            } else {
                console.log('承認メール送信完了:', info.response);
            }
        });
    });
}

// 管理者ログアウト
app.get('/admin/logout', (req, res) => {
    req.session.destroy();
    res.redirect('/admin');
});

app.listen(PORT, () => {
    console.log(`サーバーがポート${PORT}で起動しました`);
});